<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\ChatService;

class ChatController extends Controller
{
    protected $chatService;

    public function __construct(ChatService $chatService)
    {
        $this->chatService = $chatService;
    }

    /**
     * Display the chat interface
     */
    public function index()
    {
        return view('chat.index');
    }

    /**
     * Handle incoming chat messages
     */
    public function sendMessage(Request $request)
    {
        $request->validate([
            'message' => 'required|string|max:1000',
            'conversation_history' => 'array',
            'current_state' => 'string',
            'user_data' => 'array'
        ]);

        $message = $request->input('message');
        $conversationHistory = $request->input('conversation_history', []);
        $currentState = $request->input('current_state', 'WELCOME');
        $userData = $request->input('user_data', []);

        // Extract any new user data from the message using ChatService
        $newData = $this->chatService->extractUserData($message, $currentState);
        $userData = array_merge($userData, $newData);

        // new codes

        // DEBUG: Log what was extracted
\Log::info('Current State: ' . $currentState);
\Log::info('Extracted Data: ' . json_encode($newData));
\Log::info('Full User Data: ' . json_encode($userData));

// Use advanceState to determine next state
$stateMachine = $this->advanceState($currentState, $message, $userData);

// DEBUG: Log state transition
\Log::info('Next State: ' . $stateMachine['next_state']);
\Log::info('Progress: ' . $stateMachine['progress']);

        //new codes

        // Get AI response from ChatService
        $response = $this->chatService->sendMessage(
            $message,
            $conversationHistory,
            $currentState,
            $userData
        );

        if ($response['success']) {
            return response()->json([
                'success' => true,
                'message' => $response['message'],
                'next_state' => $response['next_state'],
                'state_progress' => $response['state_progress'],
                'user_data' => $userData
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => $response['message'],
                'error' => $response['error'] ?? null
            ], 500);
        }
    }

    /**
     * Reset conversation
     */
    public function resetConversation()
    {
        return response()->json([
            'success' => true,
            'message' => 'Conversation reset successfully'
        ]);
    }

    /**
     * Get quick reply suggestions based on current state
     */
    public function getQuickReplies(Request $request)
    {
        $currentState = $request->input('current_state', 'WELCOME');
        
        $quickReplies = [
            'WELCOME' => [
                "Yes, let's start!",
                "Tell me more first",
            ],
            'GARMENT_TYPE' => [
                "Top",
                "Bottom/Skirt",
                "Dress"
            ],
            'BODY_SHAPE' => [
                "Pear (fuller hips)",
                "Apple (fuller midsection)",
                "Hourglass (balanced)",
                "Rectangle (straight)"
            ],
            'FIT_PREFERENCE' => [
                "Fitted",
                "Relaxed",
                "Regular"
            ],
            'CLOSING' => [
                "Start over",
                "Thank you!"
            ]
        ];

        return response()->json([
            'quick_replies' => $quickReplies[$currentState] ?? []
        ]);
    }

    /**
     * Advanced state machine logic
     */
    private function advanceState($state, $message, $userData)
    {
        switch ($state) {
            case 'WELCOME':
                $next = 'BASIC_INFO';
                break;

            case 'BASIC_INFO':
                if (!empty($userData['height']) && !empty($userData['current_size'])) {
                    $next = 'GARMENT_TYPE';
                } else {
                    $next = 'BASIC_INFO'; // Stay until data is complete
                }
                break;

            case 'GARMENT_TYPE':
                $next = !empty($userData['garment_type']) ? 'BODY_SHAPE' : 'GARMENT_TYPE';
                break;

            case 'BODY_SHAPE':
                $next = !empty($userData['body_shape']) ? 'FIT_PREFERENCE' : 'BODY_SHAPE';
                break;

            case 'FIT_PREFERENCE':
                if (!empty($userData['fit_preference']) && !empty($userData['occasion'])) {
                    $next = 'BRAND_CONTEXT';
                } else {
                    $next = 'FIT_PREFERENCE';
                }
                break;

            case 'BRAND_CONTEXT':
                $next = 'FIT_RECOMMENDATION';
                break;

            case 'FIT_RECOMMENDATION':
                $next = 'SUSTAINABILITY_NOTE';
                break;

            case 'SUSTAINABILITY_NOTE':
                $next = 'CLOSING';
                break;

            default:
                $next = 'CLOSING';
        }

        // Progress percentage calculation
        $states = [
            'WELCOME' => 0,
            'BASIC_INFO' => 11,
            'GARMENT_TYPE' => 22,
            'BODY_SHAPE' => 33,
            'FIT_PREFERENCE' => 44,
            'BRAND_CONTEXT' => 55,
            'FIT_RECOMMENDATION' => 77,
            'SUSTAINABILITY_NOTE' => 88,
            'CLOSING' => 100,
        ];

        return [
            'next_state' => $next,
            'user_data' => $userData,
            'progress' => $states[$next] ?? 0
        ];
    }
}