<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ChatService
{
    /** FULL CONVERSATION FLOW **/
    private $conversationStates = [
        'WELCOME',
        'BASIC_INFO',
        'GARMENT_TYPE',
        'BODY_SHAPE',
        'FIT_PREFERENCE',
        'BRAND_CONTEXT',
        'FIT_RECOMMENDATION',
        'SUSTAINABILITY_NOTE',
        'CLOSING'
    ];

    /**
     * SYSTEM PROMPT BASED ON STATE
     */
    private function getSystemPrompt($state, $data)
    {
        $base = "You are FitMatch, a body-positive AI sizing assistant for Punkyz.
        You speak warmly, casually, and respectfully.
        You prioritize African and plus-size body realities.
        Always use friendly, encouraging language.";

        $prompts = [

            'WELCOME' => "Welcome the user warmly.
            Explain briefly what Punkyz FitMatch does.
            Ask if they’re ready to begin.",

            'BASIC_INFO' => "Ask politely: 
            – Their height (in words is OK, e.g. “5'6” or “170cm”)
            – Their current dress/top/bottom size.
            Reassure them that all info is private and only for sizing.",

            'GARMENT_TYPE' => "Ask what they want to shop for:
            – Top
            – Bottom/Skirt
            – Dress
            Keep it simple and friendly.",

            'BODY_SHAPE' => "Help the user identify their body shape (Pear, Apple, Hourglass, Rectangle)
            Give short descriptions for each and ask them to choose.",

           'FIT_PREFERENCE' => "Ask if they prefer:
– Fitted
– Relaxed  
– Regular
ALSO ask what occasion: casual, office, or special event.
Make it one friendly question combining both fit preference and occasion.",

            'BRAND_CONTEXT' => "Give a friendly simulated explanation like:
            “Brands differ slightly; Punkyz tends to fit true-to-size for tops but more relaxed for bottoms.”
            Then transition into calculating their size soon.",

            'FIT_RECOMMENDATION' => 
                "Using all collected data:\n"
                . "Height: " . ($data['height'] ?? 'N/A') . "\n"
                . "Size: " . ($data['current_size'] ?? 'N/A') . "\n"
                . "Garment: " . ($data['garment_type'] ?? 'N/A') . "\n"
                . "Body shape: " . ($data['body_shape'] ?? 'N/A') . "\n"
                . "Fit preference: " . ($data['fit_preference'] ?? 'N/A') . "\n"
                . "Occasion: " . ($data['occasion'] ?? 'N/A') . "\n\n"
                . "Give a size recommendation + 2 styling tips.",

            'SUSTAINABILITY_NOTE' => "Give a short sentence about how choosing the right fit helps reduce returns and environmental waste.",

            'CLOSING' => "Close warmly. Thank them. Offer to restart the process if they want."
        ];

        return $base . "\n\n" . ($prompts[$state] ?? "");
    }


    /**
     * SEND MESSAGE TO OPENAI
     */
    public function sendMessage($message, $history, $currentState, $userData)
    {
        $key = env('OPENAI_API_KEY');

        if (!$key) {
            return ['success' => false, 'message' => 'Missing API key'];
        }

        try {
            $messages = [
                ['role' => 'system', 'content' => $this->getSystemPrompt($currentState, $userData)]
            ];

            foreach ($history as $msg) {
                $messages[] = [
                    'role' => $msg['role'],
                    'content' => $msg['content']
                ];
            }

            $messages[] = ['role' => 'user', 'content' => $message];

            $response = Http::timeout(30)->withHeaders([
    'Authorization' => "Bearer $key",
    'Content-Type' => 'application/json'
])->post("https://api.openai.com/v1/chat/completions", [
    "model" => "gpt-4o-mini",
    "messages" => $messages,
    "max_tokens" => 300,
    "temperature" => 0.7
]);

            if (!$response->successful()) {
                Log::error("API ERROR: ".$response->body());
                return ['success' => false, 'message' => 'AI error', 'error' => $response->body()];
            }

            $data = $response->json();
           // $aiText = $data['output_text'] ?? "Sorry, I didn’t quite get that.";
            $aiText = $data['choices'][0]['message']['content'] ?? "Sorry, I didn't quite get that.";

            \Log::info('OpenAI Response: ' . $aiText);

            /** Determine next state */
            $next = $this->nextState($currentState, $userData);

            return [
                'success' => true,
                'message' => $aiText,
                'next_state' => $next,
                'state_progress' => $this->progressPercent($next)
            ];

        } catch (\Exception $e) {
            Log::error("ChatService exception: ".$e->getMessage());
            return ['success' => false, 'message' => 'Unexpected error', 'error' => $e->getMessage()];
        }
    }


    /**
     * STRICT STATE MACHINE
     */
    private function nextState($state, $data)
{
    switch ($state) {

        case 'WELCOME':
            // Always go to BASIC_INFO after greeting
            return 'BASIC_INFO';

        case 'BASIC_INFO':
            // Wait until both height and current size are provided
            if (!empty($data['height']) && !empty($data['current_size'])) {
                return 'GARMENT_TYPE';
            }
            return 'BASIC_INFO';

        case 'GARMENT_TYPE':
            // Wait for the user to choose the garment type
            if (!empty($data['garment_type'])) {
                return 'BODY_SHAPE';
            }
            return 'GARMENT_TYPE';

        case 'BODY_SHAPE':
            // Wait for the user to select their body shape
            if (!empty($data['body_shape'])) {
                return 'FIT_PREFERENCE';
            }
            return 'BODY_SHAPE';

        case 'FIT_PREFERENCE':
            // Wait until both fit preference and occasion are provided
            if (!empty($data['fit_preference']) && !empty($data['occasion'])) {
                return 'BRAND_CONTEXT';
            }
            return 'FIT_PREFERENCE';

        case 'BRAND_CONTEXT':
            // Automatically proceed to FIT_RECOMMENDATION after context
            return 'FIT_RECOMMENDATION';

        case 'FIT_RECOMMENDATION':
            // Automatically proceed to SUSTAINABILITY_NOTE
            return 'SUSTAINABILITY_NOTE';

        case 'SUSTAINABILITY_NOTE':
            // Automatically proceed to closing
            return 'CLOSING';

        case 'CLOSING':
        default:
            return 'CLOSING';
    }
}



    /**
     * PROGRESS
     */
    private function progressPercent($state)
    {
        $i = array_search($state, $this->conversationStates);
        if ($i === false) return 0;
        return round(($i / (count($this->conversationStates)-1)) * 100);
    }


    /**
     * EXTRACT USER DATA
     */
    public function extractUserData($msg, $state)
    {
        $d = [];

        switch ($state) {

            case 'BASIC_INFO':
                if (preg_match('/(\d+)\s*cm/i', $msg, $m)) $d['height'] = $m[1] . "cm";
                if (preg_match('/(\d+)\s*kg/i', $msg, $m)) $d['weight'] = $m[1] . "kg";
                if (preg_match('/size\s*([0-9a-zA-Z]+)/i', $msg, $m)) $d['current_size'] = $m[1];
                break;

            case 'GARMENT_TYPE':
                if (preg_match('/top|blouse|shirt/i', $msg)) $d['garment_type'] = 'Top';
                if (preg_match('/skirt|bottom|pant|trouser/i', $msg)) $d['garment_type'] = 'Bottom/Skirt';
                if (preg_match('/dress|gown/i', $msg)) $d['garment_type'] = 'Dress';
                break;

            case 'BODY_SHAPE':
                foreach (['Pear', 'Apple', 'Hourglass', 'Rectangle'] as $shape) {
                    if (stripos($msg, $shape) !== false) {
                        $d['body_shape'] = $shape;
                    }
                }
                break;

           case 'FIT_PREFERENCE':
    if (preg_match('/fitted/i', $msg)) $d['fit_preference'] = 'Fitted';
    if (preg_match('/relaxed/i', $msg)) $d['fit_preference'] = 'Relaxed';
    if (preg_match('/regular/i', $msg)) $d['fit_preference'] = 'Regular';

    // Better occasion detection
    if (preg_match('/casual/i', $msg)) $d['occasion'] = 'Casual';
    if (preg_match('/office|work|professional/i', $msg)) $d['occasion'] = 'Office';
    if (preg_match('/event|party|special/i', $msg)) $d['occasion'] = 'Event';
    
    // If user didn't specify occasion but gave fit preference, default to casual
    if (!empty($d['fit_preference']) && empty($d['occasion'])) {
        $d['occasion'] = 'Casual'; // Default
    }
    break;
        }

        return $d;
    }
}
