<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Punkyz FitMatch - AI Sizing Assistant</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@600;700&family=Inter:wght@400;500&display=swap" rel="stylesheet">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --coral: #FF6B6B;
            --blush: #FFB5B5;
            --cream: #FFF5EE;
            --warm-grey: #4A4A4A;
            --soft-teal: #88D8D8;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, var(--cream) 0%, #FFE4E1 100%);
            min-height: 100vh;
            padding: 20px 0;
        }

        h1, h2, h3, h4, h5, h6 {
            font-family: 'Poppins', sans-serif;
        }

        .chat-container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 24px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }

        .chat-header {
            background: linear-gradient(135deg, var(--coral) 0%, var(--blush) 100%);
            color: white;
            padding: 24px;
            text-align: center;
        }

        .chat-header h1 {
            font-size: 1.8rem;
            margin-bottom: 8px;
            font-weight: 700;
        }

        .chat-header p {
            margin: 0;
            opacity: 0.95;
            font-size: 0.95rem;
        }

        .progress-container {
            background: rgba(255,255,255,0.2);
            height: 8px;
            border-radius: 4px;
            margin-top: 16px;
            overflow: hidden;
        }

        .progress-bar {
            background: white;
            height: 100%;
            border-radius: 4px;
            transition: width 0.5s ease;
        }

        .chat-body {
            height: 500px;
            overflow-y: auto;
            padding: 24px;
            background: var(--cream);
        }

        .message {
            margin-bottom: 20px;
            display: flex;
            align-items: flex-start;
            animation: slideIn 0.3s ease;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .message.bot {
            justify-content: flex-start;
        }

        .message.user {
            justify-content: flex-end;
        }

        .message-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            flex-shrink: 0;
        }

        .message.bot .message-avatar {
            background: linear-gradient(135deg, var(--coral), var(--blush));
            color: white;
            margin-right: 12px;
        }

        .message.user .message-avatar {
            background: var(--soft-teal);
            color: white;
            margin-left: 12px;
        }

        .message-content {
            max-width: 70%;
            padding: 16px 20px;
            border-radius: 20px;
            line-height: 1.6;
        }

        .message.bot .message-content {
            background: white;
            color: var(--warm-grey);
            border-bottom-left-radius: 4px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .message.user .message-content {
            background: linear-gradient(135deg, var(--coral), var(--blush));
            color: white;
            border-bottom-right-radius: 4px;
        }

        .typing-indicator {
            display: none;
            padding: 16px 20px;
            background: white;
            border-radius: 20px;
            border-bottom-left-radius: 4px;
            width: fit-content;
        }

        .typing-indicator.active {
            display: block;
        }

        .typing-dot {
            display: inline-block;
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: var(--coral);
            margin: 0 2px;
            animation: typing 1.4s infinite;
        }

        .typing-dot:nth-child(2) {
            animation-delay: 0.2s;
        }

        .typing-dot:nth-child(3) {
            animation-delay: 0.4s;
        }

        @keyframes typing {
            0%, 60%, 100% {
                transform: translateY(0);
            }
            30% {
                transform: translateY(-10px);
            }
        }

        .quick-replies {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-top: 12px;
        }

        .quick-reply-btn {
            padding: 8px 16px;
            background: white;
            border: 2px solid var(--blush);
            color: var(--coral);
            border-radius: 20px;
            font-size: 0.9rem;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 500;
        }

        .quick-reply-btn:hover {
            background: var(--coral);
            color: white;
            border-color: var(--coral);
            transform: translateY(-2px);
        }

        .chat-input-container {
            padding: 20px 24px;
            background: white;
            border-top: 1px solid #f0f0f0;
        }

        .input-wrapper {
            display: flex;
            gap: 12px;
            align-items: center;
        }

        .chat-input {
            flex: 1;
            padding: 14px 20px;
            border: 2px solid #f0f0f0;
            border-radius: 24px;
            font-size: 0.95rem;
            transition: border-color 0.3s ease;
        }

        .chat-input:focus {
            outline: none;
            border-color: var(--coral);
        }

        .send-btn {
            padding: 14px 28px;
            background: linear-gradient(135deg, var(--coral), var(--blush));
            color: white;
            border: none;
            border-radius: 24px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .send-btn:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255,107,107,0.3);
        }

        .send-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .reset-btn {
            padding: 8px 16px;
            background: white;
            border: 2px solid var(--warm-grey);
            color: var(--warm-grey);
            border-radius: 20px;
            font-size: 0.85rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .reset-btn:hover {
            background: var(--warm-grey);
            color: white;
        }

        .welcome-screen {
            text-align: center;
            padding: 40px 20px;
        }

        .welcome-icon {
            font-size: 4rem;
            margin-bottom: 20px;
        }

        .welcome-screen h2 {
            color: var(--coral);
            margin-bottom: 16px;
        }

        .welcome-screen p {
            color: var(--warm-grey);
            margin-bottom: 24px;
            line-height: 1.6;
        }

        .start-btn {
            padding: 16px 40px;
            background: linear-gradient(135deg, var(--coral), var(--blush));
            color: white;
            border: none;
            border-radius: 30px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .start-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(255,107,107,0.3);
        }

        @media (max-width: 768px) {
            .chat-container {
                border-radius: 0;
                height: 100vh;
            }

            .chat-body {
                height: calc(100vh - 280px);
            }

            .message-content {
                max-width: 85%;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="chat-container">
            <!-- Chat Header -->
            <div class="chat-header">
                <h1>✨ FitMatch by Punkyz</h1>
                <p>Your AI-powered sizing assistant for the perfect fit</p>
                <div class="progress-container">
                    <div class="progress-bar" id="progressBar" style="width: 0%"></div>
                </div>
            </div>

            <!-- Chat Body -->
            <div class="chat-body" id="chatBody">
                <div class="welcome-screen" id="welcomeScreen">
                    <div class="welcome-icon">👗</div>
                    <h2>Welcome to FitMatch!</h2>
                    <p>
                        I'm here to help you find your perfect size!<br>
                        Let's celebrate your beautiful body and find clothes that make you feel amazing.
                    </p>
                    <button class="start-btn" onclick="startConversation()">
                        Let's Get Started! 🎉
                    </button>
                </div>

                <div id="messagesContainer"></div>
                
                <div class="message bot" style="display: none;">
                    <div class="message-avatar">🤖</div>
                    <div class="typing-indicator" id="typingIndicator">
                        <span class="typing-dot"></span>
                        <span class="typing-dot"></span>
                        <span class="typing-dot"></span>
                    </div>
                </div>
            </div>

            <!-- Quick Replies -->
            <div class="quick-replies px-4 pb-3" id="quickRepliesContainer"></div>

            <!-- Chat Input -->
            <div class="chat-input-container">
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <small class="text-muted">Powered by AI</small>
                    <button class="reset-btn" onclick="resetChat()">↻ Start Over</button>
                </div>
                <div class="input-wrapper">
                    <input 
                        type="text" 
                        class="chat-input" 
                        id="messageInput" 
                        placeholder="Type your message here..."
                        onkeypress="handleKeyPress(event)"
                    >
                    <button class="send-btn" id="sendBtn" onclick="sendMessage()">
                        Send 📤
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Chat JavaScript -->
    <script>
        let conversationHistory = [];
        let currentState = 'WELCOME';
        let userData = {};
        let chatStarted = false;

        // CSRF Token setup
        const csrfToken = document.querySelector('meta[name="csrf-token"]').content;

        function startConversation() {
            chatStarted = true;
            document.getElementById('welcomeScreen').style.display = 'none';
            
            // Send initial greeting
            sendInitialMessage();
        }

        async function sendInitialMessage() {
            const welcomeMessage = "Hi! I'm ready to help you find your perfect size!";
            await sendMessageToAPI(welcomeMessage);
        }

        async function sendMessage() {
            const input = document.getElementById('messageInput');
            const message = input.value.trim();

            if (!message) return;

            // Display user message
            appendMessage('user', message);
            input.value = '';

            // Show typing indicator
            showTypingIndicator();

            // Send to API
            await sendMessageToAPI(message);
        }

        async function sendMessageToAPI(message) {
            try {
              const response = await fetch('{{ url("/api/chat/send") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    },
                    body: JSON.stringify({
                        message: message,
                        conversation_history: conversationHistory,
                        current_state: currentState,
                        user_data: userData
                    })
                });

                const data = await response.json();

                // Hide typing indicator
                hideTypingIndicator();

                if (data.success) {
                    // Update conversation history
                    conversationHistory.push({ role: 'user', content: message });
                    conversationHistory.push({ role: 'assistant', content: data.message });

                    // Update state and user data
                    currentState = data.next_state;
                    userData = data.user_data;

                    // Update progress bar
                    updateProgress(data.state_progress);

                    // Display bot message
                    appendMessage('bot', data.message);

                    // Load quick replies for this state
                    loadQuickReplies();
                } else {
                    appendMessage('bot', data.message || 'Sorry, something went wrong. Please try again.');
                }
            } catch (error) {
                hideTypingIndicator();
                appendMessage('bot', 'Sorry, I\'m having trouble connecting. Please check your internet and try again.');
                console.error('Error:', error);
            }
        }

        function appendMessage(type, text) {
            const container = document.getElementById('messagesContainer');
            const messageDiv = document.createElement('div');
            messageDiv.className = `message ${type}`;

            const avatar = document.createElement('div');
            avatar.className = 'message-avatar';
            avatar.textContent = type === 'bot' ? '🤖' : '👤';

            const content = document.createElement('div');
            content.className = 'message-content';
            content.textContent = text;

            if (type === 'bot') {
                messageDiv.appendChild(avatar);
                messageDiv.appendChild(content);
            } else {
                messageDiv.appendChild(content);
                messageDiv.appendChild(avatar);
            }

            container.appendChild(messageDiv);
            scrollToBottom();
        }

        function showTypingIndicator() {
            document.getElementById('typingIndicator').classList.add('active');
            scrollToBottom();
        }

        function hideTypingIndicator() {
            document.getElementById('typingIndicator').classList.remove('active');
        }

        function scrollToBottom() {
            const chatBody = document.getElementById('chatBody');
            chatBody.scrollTop = chatBody.scrollHeight;
        }

        function updateProgress(progress) {
            document.getElementById('progressBar').style.width = progress + '%';
        }

        async function loadQuickReplies() {
            try {
                const response = await fetch(`{{ url("/api/chat/quick-replies") }}?current_state=${currentState}`);
                const data = await response.json();

                const container = document.getElementById('quickRepliesContainer');
                container.innerHTML = '';

                if (data.quick_replies && data.quick_replies.length > 0) {
                    data.quick_replies.forEach(reply => {
                        const btn = document.createElement('button');
                        btn.className = 'quick-reply-btn';
                        btn.textContent = reply;
                        btn.onclick = () => selectQuickReply(reply);
                        container.appendChild(btn);
                    });
                }
            } catch (error) {
                console.error('Error loading quick replies:', error);
            }
        }

        function selectQuickReply(text) {
            document.getElementById('messageInput').value = text;
            sendMessage();
        }

        function handleKeyPress(event) {
            if (event.key === 'Enter') {
                sendMessage();
            }
        }

        function resetChat() {
            if (confirm('Are you sure you want to start over?')) {
                conversationHistory = [];
                currentState = 'WELCOME';
                userData = {};
                chatStarted = false;

                document.getElementById('messagesContainer').innerHTML = '';
                document.getElementById('quickRepliesContainer').innerHTML = '';
                document.getElementById('welcomeScreen').style.display = 'block';
                updateProgress(0);
            }
        }

        // Disable send button when input is empty
        document.getElementById('messageInput').addEventListener('input', function() {
            const sendBtn = document.getElementById('sendBtn');
            sendBtn.disabled = this.value.trim() === '';
        });
    </script>
</body>
</html>